<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Purchase;
use App\Models\PurchaseDetail;
use App\Models\Member;
use Illuminate\Http\Request;
use Carbon\Carbon;

class PurchaseController extends Controller
{

    public function index(Request $request)
    {
        // Query dasar tanpa relasi
        $query = Purchase::query();
        // Default filter: bulan berjalan
        if (!$request->filled('tanggal_mulai') && !$request->filled('tanggal_selesai')) {
            $tanggalMulai = Carbon::now()->startOfMonth()->toDateString();
            $tanggalSelesai = Carbon::now()->endOfMonth()->toDateString();
            $request->merge([
                'tanggal_mulai' => $tanggalMulai,
                'tanggal_selesai' => $tanggalSelesai,
            ]);
        }

        // Filter tanggal
        if ($request->filled('tanggal_mulai')) {
            $query->whereDate('tanggal', '>=', $request->tanggal_mulai);
        }
        if ($request->filled('tanggal_selesai')) {
            $query->whereDate('tanggal', '<=', $request->tanggal_selesai);
        }
        // Filter pencarian
        if ($request->filled('q')) {
            $search = $request->q;
            $query->where(function ($q) use ($search) {
                $q->whereHas('member', function ($q2) use ($search) {
                    $q2->where('nama', 'like', '%'.$search.'%');
                })
                ->orWhere('nama_pembeli', 'like', '%'.$search.'%')
                ->orWhereHas('details.product', function ($q3) use ($search) {
                    $q3->where('nama', 'like', '%'.$search.'%');
                });
            });
        }

        // Untuk tampilan, baru load relasi
        $purchases = (clone $query)
            ->with(['member', 'details.product'])
            ->orderByDesc('tanggal')
            ->paginate(20)
            ->withQueryString();

        // Untuk total transaksi, cukup sum tanpa relasi
        $totalTransaksi = (clone $query)->sum('total_harga');

        return view('purchases.index', compact('purchases', 'totalTransaksi'));
    }

    public function create()
    {
        $products = Product::orderBy('nama')->get();
        return view('purchases.create', compact('products'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'produk_id' => 'required|array|min:1',
            'qty' => 'required|array',
            'qty.*' => 'required|integer|min:1',
            'member_id' => 'nullable|exists:members,id',
            'nama_pembeli' => 'nullable|required_without:member_id|string|max:100',
            'bayar' => 'required|integer|min:0',
        ]);
    
        // Hitung total harga
        $total = 0;
        foreach ($request->produk_id as $i => $pid) {
            $produk = Product::find($pid);
            $total += $produk->harga * $request->qty[$i];
        }

        // Validasi bayar cukup
        if ($request->bayar < $total) {
            return back()->withErrors(['bayar' => 'Nominal bayar kurang dari total belanja.'])->withInput();
        }

        // Simpan purchase
        $purchase = Purchase::create([
            'member_id' => $request->member_id,
            'nama_pembeli' => $request->nama_pembeli,
            'tanggal' => now(),
            'total_harga' => $total,
            'bayar' => $request->bayar,
            'kembali' => $request->bayar - $total,
        ]);

        // Simpan detail
        foreach ($request->produk_id as $i => $pid) {
            $produk = Product::find($pid);
            PurchaseDetail::create([
                'purchase_id' => $purchase->id,
                'product_id' => $pid,
                'qty' => $request->qty[$i],
                'harga_satuan' => $produk->harga,
            ]);
            // Kurangi stok
            $produk->decrement('stok', $request->qty[$i]);
        }

        // Redirect ke halaman cetak struk!
        return redirect()->route('purchases.print', $purchase->id);
    }


    public function print(Purchase $purchase)
    {
        // logic cetak (return view/cetak struk/pdf/dll)
        $purchase->load(['member', 'details.product']);
        return view('purchases.print', compact('purchase'));
    }
}